#!/bin/bash

# Vérifie si l'utilisateur est root
if [[ $EUID -ne 0 ]]; then
    echo "Ce script doit être exécuté en tant que root."
    exit 1
fi

# Vérifie si apt est disponible
if ! command -v apt >/dev/null 2>&1; then
    echo "❌ Ce script nécessite un système basé sur Debian/Ubuntu avec apt."
    exit 1
fi

# Fonction de nettoyage
cleanup() {
    echo "🧹 Nettoyage des fichiers temporaires..."
    rm -rf "$TMP_DIR" /tmp/glpi.tgz 2>/dev/null
}
trap cleanup EXIT

# === Variables configurables via saisie utilisateur ===
read -p "Entrez le port Apache souhaité (ex : 8080) : " APACHE_PORT
read -p "Entrez le nom du dossier pour GLPI (dans /var/www/html/) : " GLPI_DIR
read -p "Nom de la base de données GLPI : " DB_NAME
read -p "Nom d'utilisateur de la base de données GLPI : " DB_USER
read -s -p "Mot de passe pour l'utilisateur de la base de données : " DB_PASS
echo ""
read -p "Entrez l'hôte de la base de données (ex : localhost) : " DB_HOST
read -p "Entrez l'utilisateur root de MySQL : " DB_ROOT_USER
read -s -p "Entrez le mot de passe root de MySQL : " DB_ROOT_PASS
echo ""

# === Validation des entrées ===
if ! [[ "$APACHE_PORT" =~ ^[0-9]+$ ]] || [ "$APACHE_PORT" -lt 1 ] || [ "$APACHE_PORT" -gt 65535 ]; then
    echo "❌ Le port $APACHE_PORT n'est pas valide (doit être entre 1 et 65535)."
    exit 1
fi

if ! [[ "$GLPI_DIR" =~ ^[a-zA-Z0-9_-]+$ ]]; then
    echo "❌ Le nom du dossier GLPI doit contenir uniquement des lettres, chiffres, tirets ou underscores."
    exit 1
fi

if ! [[ "$DB_NAME" =~ ^[a-zA-Z0-9_-]+$ ]]; then
    echo "❌ Le nom de la base de données doit contenir uniquement des lettres, chiffres, tirets ou underscores."
    exit 1
fi

if ! [[ "$DB_USER" =~ ^[a-zA-Z0-9_-]+$ ]]; then
    echo "❌ Le nom de l'utilisateur de la base de données doit contenir uniquement des lettres, chiffres, tirets ou underscores."
    exit 1
fi

# === Vérification si le port est déjà utilisé ===
if netstat -tuln | grep -q ":$APACHE_PORT "; then
    echo "❌ Le port $APACHE_PORT est déjà utilisé par un autre service."
    exit 1
fi

# === Vérification si le dossier existe ===
if [[ -e "/var/www/html/$GLPI_DIR" && -n "$(ls -A /var/www/html/$GLPI_DIR)" ]]; then
    echo "❌ Le dossier /var/www/html/$GLPI_DIR existe déjà et n'est pas vide."
    exit 1
fi

# === Mise à jour du système ===
apt update && apt upgrade -y
if [[ $? -ne 0 ]]; then
    echo "❌ Échec de la mise à jour du système."
    exit 1
fi

# === Installation des dépendances ===
apt install -y apache2 mariadb-server php php-mysql php-xml php-curl php-mbstring php-gd php-imap php-ldap php-apcu php-zip php-bz2 php-intl wget unzip curl jq
if [[ $? -ne 0 ]]; then
    echo "❌ Échec de l'installation des dépendances."
    exit 1
fi

# === Activation des modules Apache nécessaires ===
a2enmod rewrite
if [[ $? -ne 0 ]]; then
    echo "❌ Échec de l'activation du module rewrite."
    exit 1
fi

# === Récupération de la dernière version GLPI depuis GitHub ===
GLPI_URL=$(curl -s https://api.github.com/repos/glpi-project/glpi/releases/latest | jq -r '.assets[] | select(.name | test("glpi.*.tgz")) | .browser_download_url')
if [[ -z "$GLPI_URL" ]]; then
    echo "❌ Impossible de récupérer l'URL de la dernière version de GLPI."
    exit 1
fi

# === Téléchargement et vérification ===
wget -O /tmp/glpi.tgz "$GLPI_URL"
if [[ ! -s /tmp/glpi.tgz ]]; then
    echo "❌ Échec du téléchargement de l'archive GLPI."
    exit 1
fi

# === Préparation du dossier cible ===
mkdir -p "/var/www/html/$GLPI_DIR"

# === Extraction propre ===
TMP_DIR=$(mktemp -d)
tar -xzf /tmp/glpi.tgz -C "$TMP_DIR"
if [[ $? -ne 0 ]]; then
    echo "❌ Échec de l'extraction de l'archive GLPI."
    exit 1
fi

if [[ ! -d "$TMP_DIR/glpi" ]]; then
    echo "❌ L'archive GLPI ne contient pas le dossier attendu."
    exit 1
fi

# === Déplacement des fichiers ===
mv "$TMP_DIR"/glpi/* "/var/www/html/$GLPI_DIR/"
mv "$TMP_DIR"/glpi/.htaccess "/var/www/html/$GLPI_DIR/"
if [[ $? -ne 0 ]]; then
    echo "❌ Échec du déplacement des fichiers GLPI."
    exit 1
fi

rm -rf "$TMP_DIR" /tmp/glpi.tgz

# === Permissions ===
chown -R www-data:www-data "/var/www/html/$GLPI_DIR"
chmod -R 755 "/var/www/html/$GLPI_DIR"
find "/var/www/html/$GLPI_DIR" -type d -name "config" -exec chmod 750 {} \;
find "/var/www/html/$GLPI_DIR" -type d -name "files" -exec chmod 750 {} \;

# === Configuration Apache ===
if [[ ! -f /etc/apache2/ports.conf ]]; then
    cat > /etc/apache2/ports.conf <<EOF
# Fichier de configuration des ports Apache
Listen 80
<IfModule ssl_module>
    Listen 443
</IfModule>
<IfModule mod_gnutls.c>
    Listen 443
</IfModule>
EOF
fi

if ! grep -q "Listen $APACHE_PORT" /etc/apache2/ports.conf; then
    echo "Listen $APACHE_PORT" >> /etc/apache2/ports.conf
else
    echo "ℹ️ Le port $APACHE_PORT est déjà configuré dans ports.conf."
fi

cat > "/etc/apache2/sites-available/glpi_$GLPI_DIR.conf" <<EOF
<VirtualHost *:$APACHE_PORT>
    DocumentRoot /var/www/html/$GLPI_DIR
    <Directory /var/www/html/$GLPI_DIR>
        Options Indexes FollowSymLinks
        AllowOverride All
        Require all granted
    </Directory>

    ErrorLog \${APACHE_LOG_DIR}/glpi_error.log
    CustomLog \${APACHE_LOG_DIR}/glpi_access.log combined
</VirtualHost>
EOF

a2ensite "glpi_$GLPI_DIR.conf"
if [[ $? -ne 0 ]]; then
    echo "❌ Échec de l'activation du site Apache."
    exit 1
fi

systemctl reload apache2
if [[ $? -ne 0 ]]; then
    echo "❌ Échec du rechargement d'Apache."
    exit 1
fi

# === Configuration base de données ===
if ! mysql -h "$DB_HOST" -u "$DB_ROOT_USER" -p"$DB_ROOT_PASS" -e "SELECT 1" >/dev/null 2>&1; then
    echo "❌ Échec de la connexion à MySQL avec les identifiants fournis."
    exit 1
fi

mysql -h "$DB_HOST" -u "$DB_ROOT_USER" -p"$DB_ROOT_PASS" <<MYSQL_SCRIPT
CREATE DATABASE IF NOT EXISTS \`$DB_NAME\` CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;
CREATE USER IF NOT EXISTS '$DB_USER'@'localhost' IDENTIFIED BY '$DB_PASS';
GRANT ALL PRIVILEGES ON \`$DB_NAME\`.* TO '$DB_USER'@'localhost';
FLUSH PRIVILEGES;
MYSQL_SCRIPT

if [[ $? -ne 0 ]]; then
    echo "❌ Échec de la configuration de la base de données."
    exit 1
fi

# === Message de fin ===
echo ""
echo "✅ Installation de GLPI terminée."
echo "🌐 Accès : http://<IP>:$APACHE_PORT/"
echo "📁 Dossier : /var/www/html/$GLPI_DIR"
echo "🗄️  Base de données : $DB_NAME"
echo "👤 Utilisateur DB : $DB_USER"