import glpi_api

URL = 'http://url-de-votre-glpi/apirest.php'
APPTOKEN = 'votre_app_token'
USERTOKEN = 'votre_user_token'

def create_computer(computer_data:dict):
    '''
    Fonction pour créer un ordinateur dans GLPI.
    :param computer_data: Dictionnaire contenant les données de l'ordinateur à créer.
    :return: None
    Cette fonction utilise l'API GLPI pour ajouter un nouvel ordinateur.
    Exemple de données pour l'ordinateur :
    {
        "name": "Ordinateur Test",
        "serial": "123456789",
        'entity': 0 (0 pour l'entité racine)
    }
    '''
    try:
       create_item('Computer', computer_data)

    except Exception as e:
        print(f"Une erreur s'est produite : {e}")

def create_monitor(monitor_data:dict):
    '''
    Fonction pour créer un moniteur dans GLPI.
    :param monitor_data: Dictionnaire contenant les données du moniteur à créer.
    :return: None
    Cette fonction utilise l'API GLPI pour ajouter un nouveau moniteur.
    Exemple de données pour le moniteur :
    {
        "name": "Moniteur Test",
        "serial": "987654321",
        'entity': 0 (0 pour l'entité racine)
    }
    '''
    try:
        create_item('Monitor', monitor_data)
    except Exception as e:
        print(f"Une erreur s'est produite lors de la création du moniteur : {e}")

def create_software(software_data:dict):
    '''
    Fonction pour créer un logiciel dans GLPI.
    :param software_data: Dictionnaire contenant les données du logiciel à créer.
    :return: None
    Cette fonction utilise l'API GLPI pour ajouter un nouveau logiciel.
    Exemple de données pour le logiciel :
    {
        "name": "Logiciel Test",
        "version": "1.0",
        'entity': 0 (0 pour l'entité racine)
    }
    '''
    try:
        create_item('Software', software_data)
    except Exception as e:
        print(f"Une erreur s'est produite lors de la création du logiciel : {e}")

def create_printer(printer_data:dict):
    '''
    Fonction pour créer une imprimante dans GLPI.
    :param printer_data: Dictionnaire contenant les données de l'imprimante à créer.
    :return: None
    Cette fonction utilise l'API GLPI pour ajouter une nouvelle imprimante.
    Exemple de données pour l'imprimante :
    {
        "name": "Imprimante Test",
        "serial": "1122334455",
        'entity': 0 (0 pour l'entité racine)
    }
    '''
    try:
        create_item('Printer', printer_data)
    except Exception as e:
        print(f"Une erreur s'est produite lors de la création de l'imprimante : {e}")

def create_license(license_data:dict):
    """
    Crée une licence dans GLPI en utilisant create_item.
    
    Args:
        license_data (dict): Dictionnaire contenant les données de la licence.
                             Champs requis : 'software_name', 'serial_number'.
                             Champs optionnels : 'entities_id' (défaut 0), 'number_exemplaries' (défaut 1).
    
    Returns:
        dict: Réponse de l'API GLPI contenant les détails de la licence créée.
    """
    try:
        # Initialiser le client GLPI API
        glpi = glpi_api.GLPI(URL, APPTOKEN, USERTOKEN)

        # Vérifier que license_data est un dictionnaire
        if not isinstance(license_data, dict) or not license_data:
            raise ValueError("license_data doit être un dictionnaire non vide")

        # Vérifier les champs requis
        if 'software_name' not in license_data or not isinstance(license_data['software_name'], str):
            raise ValueError("software_name doit être une chaîne non vide dans license_data")
        if 'serial_number' not in license_data or not isinstance(license_data['serial_number'], str):
            raise ValueError("serial_number doit être une chaîne non vide dans license_data")

        # Définir les valeurs par défaut si non présentes
        entities_id = license_data.get('entities_id', 0)
        number_exemplaries = license_data.get('number_exemplaries', 1)

        # Vérifier les types des paramètres optionnels
        if not isinstance(entities_id, int):
            raise ValueError("entities_id doit être un entier")
        if not isinstance(number_exemplaries, int) or number_exemplaries < 0:
            raise ValueError("number_exemplaries doit être un entier positif")

        # Vérifier si le logiciel existe déjà, sinon le créer
        software_list = glpi.get_all_items('Software')
        software_id = next((item['id'] for item in software_list if item['name'].lower() == license_data['software_name'].lower()), None)
        if software_id is None:
            software_data = {
                "name": license_data['software_name'],
                "entities_id": entities_id
            }
            create_item('Software', software_data)
            software_list = glpi.get_all_items('Software')  # Mettre à jour la liste
            software_id = next((item['id'] for item in software_list if item['name'].lower() == license_data['software_name'].lower()), None)
            if software_id is None:
                raise ValueError(f"Échec de la création du logiciel '{license_data['software_name']}'")

        # Préparer les données de la licence
        license_data_final = {
            "name": license_data['software_name'],
            "serial": license_data['serial_number'],
            "entities_id": entities_id,
            "number": number_exemplaries,
            "softwares_id": software_id
        }

        # Créer la licence en utilisant create_item
        response = create_item('SoftwareLicense', license_data_final)
        return response

    except Exception as e:
        print(f"Une erreur s'est produite lors de la création de la licence : {e}")
        raise

def create_network_equipment(network_equipment_data:dict):
    """
    Crée un équipement réseau dans GLPI en utilisant create_item.
    
    Args:
        network_equipment_data (dict): Dictionnaire contenant les données de l'équipement réseau.
                                      Champs requis : 'name'.
                                      Champs optionnels : 'serial_number', 'entities_id' (défaut 0), 
                                                         'locations_id' (défaut 0), 'networkequipmenttypes_id' (défaut 1).
    
    Returns:
        dict: Réponse de l'API GLPI contenant les détails de l'équipement réseau créé.
    """
    try:
        # Initialiser le client GLPI API
        glpi = glpi_api.GLPI(URL, APPTOKEN, USERTOKEN)

        # Vérifier que network_equipment_data est un dictionnaire
        if not isinstance(network_equipment_data, dict) or not network_equipment_data:
            raise ValueError("network_equipment_data doit être un dictionnaire non vide")

        # Vérifier les champs requis
        if 'name' not in network_equipment_data or not isinstance(network_equipment_data['name'], str):
            raise ValueError("name doit être une chaîne non vide dans network_equipment_data")

        # Définir les valeurs par défaut si non présentes
        entities_id = network_equipment_data.get('entities_id', 0)
        serial_number = network_equipment_data.get('serial_number', '')
        locations_id = network_equipment_data.get('locations_id', 0)
        networkequipmenttypes_id = network_equipment_data.get('networkequipmenttypes_id', 1)

        # Vérifier les types des paramètres optionnels
        if not isinstance(entities_id, int):
            raise ValueError("entities_id doit être un entier")
        if not isinstance(locations_id, int):
            raise ValueError("locations_id doit être un entier")
        if not isinstance(networkequipmenttypes_id, int):
            raise ValueError("networkequipmenttypes_id doit être un entier")
        if serial_number and not isinstance(serial_number, str):
            raise ValueError("serial_number doit être une chaîne")

        # Préparer les données finales pour l'équipement réseau
        network_equipment_data_final = {
            "name": network_equipment_data['name'],
            "serial": serial_number,
            "entities_id": entities_id,
            "locations_id": locations_id,
            "networkequipmenttypes_id": networkequipmenttypes_id
        }

        # Créer l'équipement réseau en utilisant create_item
        response = create_item('NetworkEquipment', network_equipment_data_final)
        return response

    except Exception as e:
        print(f"Une erreur s'est produite lors de la création de l'équipement réseau : {e}")
        raise

def create_phone(phone_data:dict):
    '''
    Fonction pour créer un téléphone dans GLPI.
    :param phone_data: Dictionnaire contenant les données du téléphone à créer.
    :return: None
    Cette fonction utilise l'API GLPI pour ajouter un nouveau téléphone.
    Exemple de données pour le téléphone :
    {
        "name": "Téléphone Test",
        "serial": "123456789",
        'entity': 0 (0 pour l'entité racine)
    }
    '''
    try:
        create_item('Phone', phone_data)
    except Exception as e:
        print(f"Une erreur s'est produite lors de la création du téléphone : {e}")

def create_item(item_type, item_data):
    """
    Crée un objet dans GLPI.
    
    Args:
        item_type (str): Type de l'objet (ex. 'Computer', 'Monitor', 'User').
        item_data (dict): Dictionnaire contenant les données de l'objet (ex. {'name': 'PC-Test', 'serial': 'SN123'}).
    
    Returns:
        dict: Réponse de l'API GLPI contenant les détails de l'objet créé.
    """
    try:
        # Initialiser le client GLPI API
        glpi = glpi_api.GLPI(URL, APPTOKEN, USERTOKEN)

        # Vérifier que item_type est une chaîne non vide
        if not isinstance(item_type, str) or not item_type:
            raise ValueError("item_type doit être une chaîne non vide (ex. 'Computer')")

        # Vérifier que item_data est un dictionnaire non vide
        if not isinstance(item_data, dict) or not item_data:
            raise ValueError("item_data doit être un dictionnaire non vide")

        # Ajouter l'objet dans GLPI
        response = glpi.add(item_type, item_data)
        print(f"Objet de type '{item_type}' créé avec succès : {response}")

        # Récupérer l'ID de l'objet créé et vérifier ses détails
        item_id = response[0]['id']
        created_item = glpi.get_item(item_type, item_id)
        print(f"Objet créé - ID: {created_item['id']}, Détails: {created_item}")

        return response

    except Exception as e:
        print(f"Une erreur s'est produite lors de la création de l'objet de type '{item_type}' : {e}")
        raise

def list_glpi_items(item_type):
    '''
    Fonction pour lister les éléments GLPI d'un type donné.
    :param item_type: Type d'élément GLPI (par exemple, 'Computer', 'Printer', etc.)
    :return: None
    Cette fonction récupère et affiche tous les éléments du type spécifié.
    Elle peut être utilisée pour vérifier les éléments existants dans GLPI.
    '''
    try:
        glpi = glpi_api.GLPI(URL, APPTOKEN, USERTOKEN)
        items = glpi.get_all_items(item_type)
        for item in items:
            print(f"ID: {item['id']}, Name: {item['name']}")
    except Exception as e:
        print(f"Une erreur s'est produite : {e}")

def create_ticket(ticket_data):
    """
    Crée un ticket dans GLPI en utilisant create_item.
    
    Args:
        ticket_data (dict): Dictionnaire contenant les données du ticket.
                           Champs requis : 'name', 'content'.
                           Champs optionnels : 'entities_id' (défaut 0), 'urgency' (défaut 3), 
                                              'impact' (défaut 3), 'priority' (défaut calculé), 
                                              'itilcategories_id' (défaut 0), 'users_id_recipient' (défaut 0).
    
    Returns:
        dict: Réponse de l'API GLPI contenant les détails du ticket créé.
    """
    try:
        # Initialiser le client GLPI API
        glpi = glpi_api.GLPI(URL, APPTOKEN, USERTOKEN)

        # Vérifier que ticket_data est un dictionnaire
        if not isinstance(ticket_data, dict) or not ticket_data:
            raise ValueError("ticket_data doit être un dictionnaire non vide")

        # Vérifier les champs requis
        if 'name' not in ticket_data or not isinstance(ticket_data['name'], str):
            raise ValueError("name doit être une chaîne non vide dans ticket_data")
        if 'content' not in ticket_data or not isinstance(ticket_data['content'], str):
            raise ValueError("content doit être une chaîne non vide dans ticket_data")

        # Définir les valeurs par défaut si non présentes
        entities_id = ticket_data.get('entities_id', 0)
        urgency = ticket_data.get('urgency', 3)  # 1 (très urgent) à 5 (très faible)
        impact = ticket_data.get('impact', 3)    # 1 (très élevé) à 5 (très faible)
        itilcategories_id = ticket_data.get('itilcategories_id', 0)
        users_id_recipient = ticket_data.get('users_id_recipient', 0)

        # Calculer la priorité (facultatif, basé sur urgence et impact)
        # Priorité = (urgence + impact) / 2 (approximation, ajustez selon votre configuration GLPI)
        priority = min(max((urgency + impact) // 2, 1), 5)

        # Préparer les données finales du ticket
        ticket_data_final = {
            "name": ticket_data['name'],
            "content": ticket_data['content'],
            "entities_id": entities_id,
            "urgency": urgency,
            "impact": impact,
            "priority": priority,
            "itilcategories_id": itilcategories_id,
            "users_id_recipient": users_id_recipient,
            "status": 1  # 1 = Nouveau (ajustez selon vos besoins, ex. 2 = En attente)
        }

        # Créer le ticket en utilisant create_item
        response = create_item('Ticket', ticket_data_final)
        return response

    except Exception as e:
        print(f"Une erreur s'est produite lors de la création du ticket : {e}")
        raise
#Copyright (c) 2025 Léo FRANZ