// Fonction pour basculer entre le mode clair et le mode sombre
function toggleMode() {
    const body = document.body; // Sélectionne l'élément body
    body.classList.toggle('dark-mode'); // Ajoute ou supprime la classe 'dark-mode'
    
    // Sélectionne les boutons pour le mode sombre et clair
    const darkModeButton = document.querySelector('button[onclick="toggleMode()"]');
    const lightModeButton = document.querySelectorAll('button[onclick="toggleMode()"]')[1];
    
    // Affiche ou masque les boutons en fonction du mode actuel
    if (body.classList.contains('dark-mode')) {
        darkModeButton.style.display = 'none'; // Cache le bouton pour activer le mode sombre
        lightModeButton.style.display = 'block'; // Affiche le bouton pour activer le mode clair
    } else {
        darkModeButton.style.display = 'block'; // Affiche le bouton pour activer le mode sombre
        lightModeButton.style.display = 'none'; // Cache le bouton pour activer le mode clair
    }
}

// Fonction pour calculer le masque en fonction du type d'adresse IP (IPv4 ou IPv6)
function calculateMask() {
    // Récupère le type d'adresse IP sélectionné (IPv4 ou IPv6)
    const ipType = document.querySelector('input[name="ipType"]:checked').value;
    // Récupère l'adresse IP saisie par l'utilisateur
    const ipAddress = document.getElementById('ipAddress').value.trim();
    // Récupère le masque saisi par l'utilisateur
    const maskInput = document.getElementById('mask').value;

    // Récupère les éléments HTML pour afficher les résultats
    const resultDirect = document.getElementById('resultDirect');
    const resultInverse = document.getElementById('resultInverse');
    const resultNetworkHeader = document.getElementById('resultNetworkHeader');
    const resultNetworkAddress = document.getElementById('resultNetworkAddress');
    const resultFirstAddress = document.getElementById('resultFirstAddress');
    const resultLastAddress = document.getElementById('resultLastAddress');
    const resultBroadcastAddress = document.getElementById('resultBroadcastAddress');
    const resultAvailableIPsNetwork = document.getElementById('resultAvailableIPsNetwork');
    const resultFilterHeader = document.getElementById('resultFilterHeader');
    const resultFilterFirstAddress = document.getElementById('resultFilterFirstAddress');
    const resultFilterLastAddress = document.getElementById('resultFilterLastAddress');
    const resultAvailableIPsFilter = document.getElementById('resultAvailableIPsFilter');

    // Réinitialise les résultats affichés
    resultNetworkHeader.style.display = 'none';
    resultNetworkAddress.textContent = 'Adresse de réseau: ';
    resultFirstAddress.textContent = 'Première adresse: ';
    resultLastAddress.textContent = 'Dernière adresse: ';
    resultBroadcastAddress.textContent = 'Adresse de broadcast: ';
    resultAvailableIPsNetwork.textContent = 'Nombre d\'adresses IP disponibles: ';
    resultFilterHeader.style.display = 'none';
    resultFilterFirstAddress.textContent = 'Première adresse: ';
    resultFilterLastAddress.textContent = 'Dernière adresse: ';
    resultAvailableIPsFilter.textContent = 'Nombre d\'adresses IP disponibles: ';

    // Vérifie le type d'adresse IP et appelle la fonction correspondante
    if (ipType === 'ipv4') {
        // Appelle la fonction pour calculer les informations pour une adresse IPv4
        calculateIPv4Mask(
            ipAddress,
            maskInput,
            resultDirect,
            resultInverse,
            resultNetworkHeader,
            resultNetworkAddress,
            resultFirstAddress,
            resultLastAddress,
            resultBroadcastAddress,
            resultAvailableIPsNetwork,
            resultFilterHeader,
            resultFilterFirstAddress,
            resultFilterLastAddress,
            resultAvailableIPsFilter
        );
    } else {
        // Appelle la fonction pour calculer les informations pour une adresse IPv6
        calculateIPv6Mask(ipAddress, maskInput, resultDirect, resultInverse);
    }
}

/**
 * Calcule les informations réseau basées sur une adresse IPv4 et un masque de sous-réseau.
 *
 * @param {string} ip - L'adresse IPv4 à analyser (ex: "192.168.1.1").
 * @param {number} mask - Le masque de sous-réseau en notation CIDR (ex: 24).
 * @param {HTMLElement} resultDirect - Élément HTML pour afficher le masque direct.
 * @param {HTMLElement} resultInverse - Élément HTML pour afficher le masque inverse (wildcard).
 * @param {HTMLElement} resultNetworkHeader - Élément HTML pour afficher l'en-tête des informations réseau.
 * @param {HTMLElement} resultNetworkAddress - Élément HTML pour afficher l'adresse réseau.
 * @param {HTMLElement} resultFirstAddress - Élément HTML pour afficher la première adresse utilisable.
 * @param {HTMLElement} resultLastAddress - Élément HTML pour afficher la dernière adresse utilisable.
 * @param {HTMLElement} resultBroadcastAddress - Élément HTML pour afficher l'adresse de broadcast.
 * @param {HTMLElement} resultAvailableIPsNetwork - Élément HTML pour afficher le nombre d'adresses IP disponibles en mode réseau.
 * @param {HTMLElement} resultFilterHeader - Élément HTML pour afficher l'en-tête des informations en mode filtre.
 * @param {HTMLElement} resultFilterFirstAddress - Élément HTML pour afficher la première adresse en mode filtre.
 * @param {HTMLElement} resultFilterLastAddress - Élément HTML pour afficher la dernière adresse en mode filtre.
 * @param {HTMLElement} resultAvailableIPsFilter - Élément HTML pour afficher le nombre d'adresses IP disponibles en mode filtre.
 *
 * @throws {Error} Affiche des messages d'erreur dans les éléments HTML si le masque ou l'adresse IP est invalide.
 *
 * @description
 * Cette fonction valide l'adresse IPv4 et le masque, puis calcule les informations suivantes :
 * - Masque direct et masque inverse (wildcard).
 * - Adresse réseau.
 * - Première et dernière adresse utilisable.
 * - Adresse de broadcast.
 * - Nombre d'adresses IP disponibles en mode réseau et en mode filtre.
 *
 * Les résultats sont affichés dans les éléments HTML fournis en paramètres.
 */
function calculateIPv4Mask(
  ip,
  mask,
  resultDirect,
  resultInverse,
  resultNetworkHeader,
  resultNetworkAddress,
  resultFirstAddress,
  resultLastAddress,
  resultBroadcastAddress,
  resultAvailableIPsNetwork,
  resultFilterHeader,
  resultFilterFirstAddress,
  resultFilterLastAddress,
  resultAvailableIPsFilter
) {
  //Validation du masque (doit être un nombre entre 8 et 32)
  if (!/^\d+$/.test(mask) || mask < 8 || mask > 32) {
    resultDirect.textContent = 'Erreur: Le masque doit être un nombre entre 8 et 32';
    resultDirect.style.color = 'red'; // Change la couleur du texte en rouge pour l'erreur
    resultInverse.textContent = '';
    return;
  }
  // Validation de l'adresse IP (doit être une adresse IPv4 valide)
  const ipRegex = /^(\d{1,3})\.(\d{1,3})\.(\d{1,3})\.(\d{1,3})$/;
  if (!ipRegex.test(ip)) { // Vérification de la structure de l'IP
    // Si l'IP ne correspond pas au format, afficher une erreur
    resultDirect.textContent = 'Erreur: Adresse IPv4 invalide (ex: 192.168.1.1)';
    resultDirect.style.color = 'red'; // Change la couleur du texte en rouge pour l'erreur
    resultInverse.textContent = '';
    return;
  }
  const octets = ip.split('.').map(Number);
  if (octets.some(o => o < 0 || o > 255)) { // Vérification que chaque octet est entre 0 et 255
    // Si un octet est en dehors de la plage, afficher une erreur
    resultDirect.textContent = 'Erreur: Chaque octet doit être entre 0 et 255';
    resultDirect.style.color = 'red'; // Change la couleur du texte en rouge pour l'erreur
    resultInverse.textContent = '';
    return;
  }

  const maskBits = parseInt(mask); // Convertir le masque en bits
  let binary = '';
  let wildcard = '';
  for (let i = 0; i < 32; i++) { // Créer la représentation binaire du masque
    binary += i < maskBits ? '1' : '0';
    wildcard += i < maskBits ? '0' : '1';
  }

  const binaryGroups = binary.match(/.{1,8}/g); // Diviser en groupes de 8 bits
  const wildcardGroups = wildcard.match(/.{1,8}/g); 
  const decimalMask = binaryGroups.map(bin => parseInt(bin, 2)).join('.');
  const decimalWildcard = wildcardGroups.map(bin => parseInt(bin, 2)).join('.');

  // Calculate network address
  const ipBinary = octets.map(o => o.toString(2).padStart(8, '0')).join('');
  const networkBinary = ipBinary.split('').map((bit, i) => binary[i] === '1' ? bit : '0').join('');
  const networkGroups = networkBinary.match(/.{1,8}/g);
  const networkAddress = networkGroups.map(bin => parseInt(bin, 2)).join('.');

  // Calculate total number of addresses
  const totalAddresses = Math.pow(2, 32 - maskBits);

  // Mode réseau
  resultNetworkHeader.style.display = 'block';
  resultNetworkAddress.textContent = `Adresse de réseau: ${networkAddress}`;

  // Première adresse (réseau + 1)
// Calcul de la première adresse utilisable (adresse réseau + 1)
let networkInt = networkGroups.map(bin => parseInt(bin, 2)); // Convertit les groupes binaires en entiers
networkInt[3] += 1; // Incrémente le dernier octet
if (networkInt[3] > 255) { // Si le dernier octet dépasse 255, on effectue un report
    networkInt[3] = 0;
    networkInt[2] += 1;
    if (networkInt[2] > 255) { // Si le troisième octet dépasse 255, on effectue un report
        networkInt[2] = 0;
        networkInt[1] += 1;
        if (networkInt[1] > 255) { // Si le deuxième octet dépasse 255, on effectue un report
            networkInt[1] = 0;
            networkInt[0] += 1; // Incrémente le premier octet si nécessaire
        }
    }
}
const firstAddress = networkInt.join('.'); // Recompose l'adresse IPv4
resultFirstAddress.textContent = `Première adresse: ${firstAddress}`; // Affiche la première adresse utilisable

// Calcul de l'adresse de broadcast (adresse réseau OU wildcard)
const broadcastBinary = ipBinary.split('').map((bit, i) => wildcard[i] === '1' ? '1' : bit).join('');
const broadcastGroups = broadcastBinary.match(/.{1,8}/g); // Divise en groupes de 8 bits
const broadcastAddress = broadcastGroups.map(bin => parseInt(bin, 2)).join('.'); // Convertit en adresse IPv4
resultBroadcastAddress.textContent = `Adresse de broadcast: ${broadcastAddress}`; // Affiche l'adresse de broadcast

// Calcul de la dernière adresse utilisable (adresse de broadcast - 1)
let broadcastInt = broadcastGroups.map(bin => parseInt(bin, 2)); // Convertit les groupes binaires en entiers
broadcastInt[3] -= 1; // Décrémente le dernier octet
if (broadcastInt[3] < 0) { // Si le dernier octet est inférieur à 0, on effectue un emprunt
    broadcastInt[3] = 255;
    broadcastInt[2] -= 1;
    if (broadcastInt[2] < 0) { // Si le troisième octet est inférieur à 0, on effectue un emprunt
        broadcastInt[2] = 255;
        broadcastInt[1] -= 1;
        if (broadcastInt[1] < 0) { // Si le deuxième octet est inférieur à 0, on effectue un emprunt
            broadcastInt[1] = 255;
            broadcastInt[0] -= 1; // Décrémente le premier octet si nécessaire
        }
    }
}
const lastAddress = broadcastInt.join('.'); // Recompose l'adresse IPv4
resultLastAddress.textContent = `Dernière adresse: ${lastAddress}`; // Affiche la dernière adresse utilisable

// Calcul du nombre d'adresses IP disponibles (en mode réseau: total - 2 pour réseau et broadcast)
const availableIPsNetwork = totalAddresses - 2; // Exclut l'adresse réseau et l'adresse de broadcast
resultAvailableIPsNetwork.textContent = `Nombre d'adresses IP disponibles: ${availableIPsNetwork > 0 ? availableIPsNetwork : 0}`; // Affiche le nombre d'adresses disponibles

// Mode filtre (utilise toute la plage, pas de distinction réseau/broadcast)
resultFilterHeader.style.display = 'block'; // Affiche l'en-tête du mode filtre
const filterFirstAddress = networkAddress; // La première adresse en mode filtre est l'adresse réseau
resultFilterFirstAddress.textContent = `Première adresse: ${filterFirstAddress}`; // Affiche la première adresse en mode filtre
resultFilterLastAddress.textContent = `Dernière adresse: ${broadcastAddress}`; // La dernière adresse en mode filtre est l'adresse de broadcast
resultAvailableIPsFilter.textContent = `Nombre d'adresses IP disponibles: ${totalAddresses}`; // Affiche le nombre total d'adresses disponibles

// Affichage des masques
resultDirect.textContent = `Masque Direct: ${decimalMask} (/${maskBits})`; // Affiche le masque direct
resultInverse.textContent = `Masque Inverse (Wildcard): ${decimalWildcard}`; // Affiche le masque inverse (wildcard)
}
