using Microsoft.EntityFrameworkCore;
using ToDos.Data.Entities;
using ToDos.Data.Models;

namespace ToDos.Data.Repositories;

public class TodoRepository : ITodoRepository
{
    private readonly AppDbContext _context;

    public TodoRepository(AppDbContext context)
    {
        _context = context;
    }

    public async Task<List<TodoEntity>> GetTodosAsync(TodoQuery filter)
    {
        IQueryable<TodoEntity> query = _context.Todos;

        if (!string.IsNullOrWhiteSpace(filter.Search))
        {
            var s = filter.Search.Trim();
            query = query.Where(t =>
                t.Libelle.Contains(s) ||
                (t.Commentaire != null && t.Commentaire.Contains(s)));
        }

        if (filter.IsDone.HasValue)
        {
            query = filter.IsDone.Value
                ? query.Where(t => t.Date_realisation != null)
                : query.Where(t => t.Date_realisation == null);
        }

        if (filter.PlanifFrom.HasValue)
            query = query.Where(t => t.Date_planif >= filter.PlanifFrom.Value.Date);

        if (filter.PlanifTo.HasValue)
            query = query.Where(t => t.Date_planif <= filter.PlanifTo.Value.Date);

        if (filter.DoneFrom.HasValue)
            query = query.Where(t => t.Date_realisation >= filter.DoneFrom.Value.Date);

        if (filter.DoneTo.HasValue)
            query = query.Where(t => t.Date_realisation <= filter.DoneTo.Value.Date);

        return await query.OrderBy(t => t.Id).ToListAsync();
    }

    public Task<TodoEntity?> GetByIdAsync(int id) => _context.Todos.FindAsync(id).AsTask();

    public async Task AddAsync(TodoEntity entity)
    {
        _context.Todos.Add(entity);
        await _context.SaveChangesAsync();
    }

    public async Task UpdateAsync(TodoEntity entity)
    {
        var existing = await _context.Todos.FindAsync(entity.Id);
        if (existing is null) return;

        existing.Libelle = entity.Libelle;
        existing.Commentaire = entity.Commentaire;
        existing.Date_planif = entity.Date_planif;
        existing.Date_realisation = entity.Date_realisation;

        await _context.SaveChangesAsync();
    }

    public async Task DeleteAsync(int id)
    {
        var existing = await _context.Todos.FindAsync(id);
        if (existing is null) return;

        _context.Todos.Remove(existing);
        await _context.SaveChangesAsync();
    }
}
